/* --------------------------------------------------------------- */
/* (C) Copyright 2001,2006,                                        */
/* International Business Machines Corporation,                    */
/*                                                                 */
/* All Rights Reserved.                                            */
/* --------------------------------------------------------------- */
/* PROLOG END TAG zYx                                              */
#ifndef _MACK_H_
#define _MACK_H_

#include <stdio.h>
#include <stdlib.h>

/*
 * This is just a simplification macro used to check for failed status and
 * print the error out.
 */
#define ERRCHK(func, err)                                       \
do {                                                            \
    if (rc != DACS_SUCCESS) {                                   \
        fprintf(stderr, "%s: %s\n", func, dacs_strerror(err));  \
        exit(err);                                              \
    }                                                           \
} while(0)

/*
 * Another simplification macro, do a flush after every print in an attempt to
 * order them somewhat.  This is still not without fault, the PPE prints on
 * behalf of the SPE, so messages inevitably get messed up due to timing
 * issues.
 */
#define PRINTF(...) \
    printf(__VA_ARGS__);    \
    fflush(stdout);

/*
 * Maximum number of SPE children that can be reserved.  This is used for
 * sizing static data structures.
 */
#define MAX_NCHILD 16

/*
 * Stream IDs are used for associating particular messages that are sent and
 * received.  This allows for multiple outstanding message threads to exist
 * without interference.  
 *
 * It is also possible to have a single receiver accept (listen to) multiple 
 * message streams by using the DACS_STREAM_ALL stream ID. 
 * 
 * For our purposes, we want a unique 1:1 message stream between the parent
 * and child, so we will assign a unique ID of 1.
 */
#define HELLO_STREAM_ID 1

/*
 * This is the shared memory data layout.  It consists of a counter that is
 * used to track the last index used in the "places" array.  It also contains
 * a per-SPE array of places to say hello to.  Each SPE is assigned an index,
 * that is used to index into this array.
 *
 * The data is intentionally aligned on a 128 byte boundary, although 16 byte
 * would also work.  In practice, it is better to align data on 128 boundaries
 * for performance reasons. 
 */
typedef struct hello_shared_data
{
    unsigned int last_place;
    char place_array[MAX_NCHILD][16] __attribute__ ((aligned(16)));
} hello_shared_data_t __attribute__ ((aligned(128)));

/*
 * These are global variables that are commonly used within the PPE and SPE
 * applications.
 *
 * hello_shared_mem - On the PPE, this is the remote shared memory that is
 *                    shared with the children.
 *                    On the SPE, this is the local buffer used for DMAing the
 *                    remote shared memory in/out.
 *
 * hello_message    - On the PPE, this is the send staging buffer used for 
 *                    building  the hello message to be sent to the child.
 *                    On the SPE, this is the buffer used to receive the hello
 *                    message from the parent.
 *
 * Alignment is critical for both of these pieces of data.  Both send/receive
 * and DMA have strict alignment restrictions.  Once again, we would have been
 * safe aligning these on a 16 byte boundary, but 128 byte was used for 
 * possible performance benefits.
 *
 * Shared memory regions can be allocated on any alignement, that is not an
 * error.  However, care must be taken to insure that proper offsets because
 * the actual DMA's require the target and destination buffers to have the
 * same 16 byte alignment.
 */
hello_shared_data_t hello_shared_mem  __attribute__ ((aligned(128)));
char hello_message[80] __attribute__ ((aligned(128)));

#endif
