/* --------------------------------------------------------------  */
/* (C)Copyright 2001,2007,                                         */
/* International Business Machines Corporation,                    */
/* Sony Computer Entertainment, Incorporated,                      */
/* Toshiba Corporation,                                            */
/*                                                                 */
/* All Rights Reserved.                                            */
/*                                                                 */
/* Redistribution and use in source and binary forms, with or      */
/* without modification, are permitted provided that the           */
/* following conditions are met:                                   */
/*                                                                 */
/* - Redistributions of source code must retain the above copyright*/
/*   notice, this list of conditions and the following disclaimer. */
/*                                                                 */
/* - Redistributions in binary form must reproduce the above       */
/*   copyright notice, this list of conditions and the following   */
/*   disclaimer in the documentation and/or other materials        */
/*   provided with the distribution.                               */
/*                                                                 */
/* - Neither the name of IBM Corporation nor the names of its      */
/*   contributors may be used to endorse or promote products       */
/*   derived from this software without specific prior written     */
/*   permission.                                                   */
/*                                                                 */
/* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND          */
/* CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,     */
/* INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF        */
/* MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE        */
/* DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR            */
/* CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,    */
/* SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT    */
/* NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;    */
/* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)        */
/* HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN       */
/* CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR    */
/* OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,  */
/* EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.              */
/* --------------------------------------------------------------  */
/* PROLOG END TAG zYx                                              */

#ifndef _barrier_heavy_h_
#define _barrier_heavy_h_

#include <spu_mfcio.h>

/* function: _barrier_heavy(ea, id, lsbuf, total)
 *
 * This function implements a specialized barrier function that is robust
 * and ensures that all parties leave the barrier at as close to the same
 * time as possible.
 *
 * The barrier uses a system memory cache-line buffers 'ea'.
 * The ea buffer contains an array flags that are set by each
 * of the SPEs when they enter the barrier. 
 *
 * SPE with an id of 0, is consider the master. He waits until all the
 * flags are set by the slave SPEs, and then clears the flags to releases 
 * the slave SPEs.
 *
 * 'total' is the number of SPEs participating in the barrier.
 * 
 * Restrictions: 'ea' must point to a 16-byte aligned address in system
 * memory.  No other variables should reside in this cacheline. The 
 * 'lsbuf' must point to a 128-byte aligned address in local store memory.
 */


static __inline void _barrier_heavy(unsigned int ea, unsigned int id, volatile void *lsbuf, unsigned int total)
{
  int i;
  unsigned int flag;
  unsigned int mask;
  volatile unsigned int *ls_ptr;

  ls_ptr = (volatile unsigned int *)lsbuf;

  /* Save the callers tag mask */
  mask = spu_readch(MFC_RdTagMask);
  spu_writech(MFC_WrTagMask, 1);

  if (id == 0) {
    /* Master SPE */

    /* Wait for all the slave SPEs to enter the barrier. */
    do {
      spu_mfcdma32(lsbuf, ea, 128, 0, MFC_GETLLAR_CMD);
      (void)spu_readch(MFC_RdAtomicStat);

      for (i=1, flag=1; i<(int)total; i++) flag &= ls_ptr[i];

    } while (flag == 0);

    /* Clear the flags for all the slave SPEs. */
    for (i=1; i<(int)total; i++) ls_ptr[i] = 0;
    
    spu_mfcdma32(lsbuf, ea, 128, 0, MFC_PUT_CMD);
       
    /* Read the buffer to affect an equivalent delay 
     * on the master SPE.
     */
    spu_mfcdma32(lsbuf, ea, 128, 0, MFC_GETB_CMD);
    spu_mfcstat(MFC_TAG_UPDATE_ALL);

  } else {
    /* Slave SPE */

    /* Write to its flag word to signal that it has reached the barrier */
    ls_ptr[id] = 1;
    spu_mfcdma32(&ls_ptr[id], ea + id*sizeof(unsigned int), 4, 0, MFC_PUT_CMD);
    spu_mfcstat(MFC_TAG_UPDATE_ALL);

    do {
      spu_mfcdma32(lsbuf, ea, 128, 0, MFC_GETLLAR_CMD);
      (void)spu_readch(MFC_RdAtomicStat);
    } while (ls_ptr[id]);
  }

  /* Restore the callers tag mask */
  spu_writech(MFC_WrTagMask, mask);
}    

#endif /* _barrier_heavy_h_ */
