/***********************************************************
*  --- OpenSURF ---                                        *
*  This library is distributed under the GNU GPL. Please   *
*  contact chris.evans@irisys.co.uk for more information.  *
*                                                          *
*  C. Evans, Research Into Robust Visual Features,         *
*  MSc University of Bristol, 2008.                        *
*                                                          *
************************************************************/

/**
   NOTE: This is a hacked version of Chris Evans's OpenSURF library. It
   was hacked by Manu Viswanathan for use at USC's iLab. Please do not
   bother Chris Evans with problems in this hacked version.

   Some of the changes from the original OpenSURF code base include:
      - file name changes
      - wrapping of all classes and functions in a namespace
      - autotools build system and "librarification" for GNU/Linux
      - class/function name changes
      - minor changes to class/function semantics and external API
      - code clean-up to eliminate compiler warnings
      - etc.

   The actual guts of the implementation/internals performing the SURF
   computations remains the same.

   Nonetheless, do not trust this code to actually do what it purports to
   do. Neither Manu Viswanathan nor iLab assume any responsibility or
   liability of any sort whatsoever.
*/

//------------------------------ HEADERS --------------------------------

// OpenSURF headers
#include "config.h"
#include "opensurf.hh"
#include "Integral.hh"
#include "FastHessian.hh"
#include "SurfDescriptor.hh"

// OpenCV headers
#ifdef OPENSURF_HAVE_OPENCV
   #include <opencv/cv.h>
#else
   #error "Sorry, libopensurf needs OpenCV"
#endif

//----------------------------- NAMESPACE -------------------------------

namespace opensurf {

//--------------------------- EXTERNAL API ------------------------------

// Compute SURF keypoints and descriptors
std::vector<IPoint>
doSurf(IplImage* input_image,
       bool  rotation_invariant,
       int   num_octaves,
       int   intervals_per_octave,
       int   initial_sampling_step,
       float blob_response_threshold)
{
   std::vector<IPoint> keypoints ;

   IplImage* integral_image = Integral(input_image) ;
   FastHessian fh(integral_image, keypoints,
                  num_octaves, intervals_per_octave,
                  initial_sampling_step, blob_response_threshold) ;
   fh.getIPoints() ;

   SurfDescriptor D(integral_image, keypoints) ;
   D.getDescriptors(rotation_invariant) ;

   cvReleaseImage(& integral_image) ;

   return keypoints ;
}

// Compute only the SURF keypoints
std::vector<IPoint>
computeSurfKeypoints(IplImage* input_image,
                     int   num_octaves,
                     int   intervals_per_octave,
                     int   initial_sampling_step,
                     float blob_response_threshold)
{
   std::vector<IPoint> keypoints ;

   IplImage* integral_image = Integral(input_image) ;
   FastHessian fh(integral_image, keypoints,
                  num_octaves, intervals_per_octave,
                  initial_sampling_step, blob_response_threshold) ;
   fh.getIPoints() ;

   cvReleaseImage(& integral_image) ;

   return keypoints ;
}

// Compute the SURF descriptors given the keypoints
void
computeSurfDescriptors(IplImage* input_image,
                       std::vector<IPoint>& keypoints,
                       bool rotation_invariant)
{
   IplImage* integral_image = Integral(input_image) ;

   SurfDescriptor D(integral_image, keypoints) ;
   D.getDescriptors(rotation_invariant) ;

   cvReleaseImage(& integral_image) ;
}

//-----------------------------------------------------------------------

} // end of namespace encapsulating this file
