/***********************************************************
*  --- OpenSURF ---                                        *
*  This library is distributed under the GNU GPL. Please   *
*  contact chris.evans@irisys.co.uk for more information.  *
*                                                          *
*  C. Evans, Research Into Robust Visual Features,         *
*  MSc University of Bristol, 2008.                        *
*                                                          *
************************************************************/

/**
   NOTE: This is a hacked version of Chris Evans's OpenSURF library. It
   was hacked by Manu Viswanathan for use at USC's iLab. Please do not
   bother Chris Evans with problems in this hacked version.

   Some of the changes from the original OpenSURF code base include:
      - file name changes
      - wrapping of all classes and functions in a namespace
      - autotools build system and "librarification" for GNU/Linux
      - class/function name changes
      - minor changes to class/function semantics and external API
      - code clean-up to eliminate compiler warnings
      - etc.

   The actual guts of the implementation/internals performing the SURF
   computations remains the same.

   Nonetheless, do not trust this code to actually do what it purports to
   do. Neither Manu Viswanathan nor iLab assume any responsibility or
   liability of any sort whatsoever.
*/

//------------------------------ HEADERS --------------------------------

// OpenSURF headers
#include "config.h"
#include "IPoint.hh"

// OpenCV headers
#ifdef OPENSURF_HAVE_OPENCV
   #include <opencv/cv.h>
#else
   #error "Sorry, libopensurf needs OpenCV"
#endif

// Standard C++ headers
#include <limits>

// Standard C headers
#include <math.h>

//----------------------------- NAMESPACE -------------------------------

namespace opensurf {

//-------------------- INITIALIZATION AND CLEAN-UP ----------------------

IPoint::IPoint()
   : orientation(0)
{}

IPoint::~IPoint(){}

//-------------------- IPoint ARITHMETIC OPERATORS ----------------------

// Quick helper to return the square of a number
static inline float square(float f)
{
   return f * f ;
}

// Get the distance in descriptor space between IPoints
float IPoint::operator-(const IPoint& rhs) const
{
   float sum = 0 ;
   for (int i = 0; i < 64; ++i)
      sum += square(descriptor[i] - rhs.descriptor[i]) ;

   return sqrtf(sum) ;
}

//-------------------------- POINT MATCHING -----------------------------

// Find matches between two lists of IPoints
std::vector<MatchingIPoints>
findMatches(std::vector<IPoint>& P1, std::vector<IPoint>& P2)
{
   std::vector<MatchingIPoints> matches ;

   for (unsigned int i = 0; i < P1.size(); ++i)
   {
      float d1 = std::numeric_limits<float>::max() ;
      float d2 = std::numeric_limits<float>::max() ;
      IPoint* match = 0 ;
      for (unsigned int j = 0; j < P2.size(); ++j)
      {
         float d = P1[i] - P2[j] ;
         if (d < d1) // if this feature matches better than current best
         {
            d2 = d1 ;
            d1 = d ;
            match = & P2[j] ;
         }
         else if (d < d2) // this feature matches better than second best
         {
            d2 = d ;
         }
      }

      // If match has a d1:d2 ratio < 0.65 ipoints are a match
      if (d1/d2 < 0.65f)
      {
         // Store the change in position
         P1[i].dx = match->x - P1[i].x ;
         P1[i].dy = match->y - P1[i].y ;
         matches.push_back(MatchingIPoints(P1[i], *match)) ;
      }
   }
   return matches ;
}

// Find homography between matched points and translate src_corners to
// dst_corners.
#ifdef OPENSURF_HAVE_OPENCV_1_1

int
translateCorners(std::vector<MatchingIPoints>& matches,
                 const CvPoint src_corners[4], CvPoint dst_corners[4])
{
  int n = static_cast<int>(matches.size()) ;
  if (n < 4)
     return 0 ;

  std::vector<CvPoint2D32f> pt1, pt2 ;
  pt1.resize(n);
  pt2.resize(n);

  // Copy Ipoints from match vector into cvPoint vectors
  for (int i = 0; i < n; ++i)
  {
     pt1[i] = cvPoint2D32f(matches[i].second.x, matches[i].second.y);
     pt2[i] = cvPoint2D32f(matches[i].first.x,  matches[i].first.y);
  }

  // Find the homography (transformation) between the two sets of points
  double h[9] ;
  CvMat _h   = cvMat(3, 3, CV_64F, h) ;
  CvMat _pt1 = cvMat(1, n, CV_32FC2, & pt1[0]) ;
  CvMat _pt2 = cvMat(1, n, CV_32FC2, & pt2[0]) ;
  if (! cvFindHomography(&_pt1, &_pt2, &_h, CV_RANSAC, 5))
    return 0 ;

  // Translate src_corners to dst_corners using homography
  for (int i = 0; i < 4; ++i)
  {
     double x = src_corners[i].x, y = src_corners[i].y ;
     double Z = 1/(h[6]*x + h[7]*y + h[8]) ;
     double X = (h[0]*x + h[1]*y + h[2]) * Z ;
     double Y = (h[3]*x + h[4]*y + h[5]) * Z ;
     dst_corners[i] = cvPoint(cvRound(X), cvRound(Y)) ;
  }
  return 1 ;
}

#else // OpenCV version < 1.1

int translateCorners(std::vector<MatchingIPoints>&,
                     const CvPoint[4], CvPoint [4])
{
   return 0 ;
}

#endif

//-----------------------------------------------------------------------

} // end of namespace encapsulating this file
