/***********************************************************
*  --- OpenSURF ---                                        *
*  This library is distributed under the GNU GPL. Please   *
*  contact chris.evans@irisys.co.uk for more information.  *
*                                                          *
*  C. Evans, Research Into Robust Visual Features,         *
*  MSc University of Bristol, 2008.                        *
*                                                          *
************************************************************/

/**
   NOTE: This is a hacked version of Chris Evans's OpenSURF library. It
   was hacked by Manu Viswanathan for use at USC's iLab. Please do not
   bother Chris Evans with problems in this hacked version.

   Some of the changes from the original OpenSURF code base include:
      - file name changes
      - wrapping of all classes and functions in a namespace
      - autotools build system and "librarification" for GNU/Linux
      - class/function name changes
      - minor changes to class/function semantics and external API
      - code clean-up to eliminate compiler warnings
      - etc.

   The actual guts of the implementation/internals performing the SURF
   computations remains the same.

   Nonetheless, do not trust this code to actually do what it purports to
   do. Neither Manu Viswanathan nor iLab assume any responsibility or
   liability of any sort whatsoever.
*/

#ifndef OPENSURF_SURF_DESCRIPTOR_DOT_HH
#define OPENSURF_SURF_DESCRIPTOR_DOT_HH

//------------------------------ HEADERS --------------------------------

// OpenSURF headers
#include "Integral.hh"
#include "IPoint.hh"

// Standard C++ headers
#include <vector>

// Standard C headers
#include <math.h>

//----------------------- FORWARD DECLARATIONS --------------------------

// This should be available from the OpenCV headers
typedef struct _IplImage IplImage ;

//----------------------------- NAMESPACE -------------------------------

/**
   \namespace opensurf
   \brief Namespace encapsulating the OpenSURF library.

   All the OpenSURF classes, functions and other definitions are wrapped
   inside of the opensurf namespace.

   \remark This namespace encapsulation is specific to Manu Viswanathan's
   hacked version of the OpenSURF library and is not part of the
   original.
*/
namespace opensurf {

//------------------------- CLASS DEFINITION ----------------------------

class SurfDescriptor {
   /// Integral image where keypoints have been detected
   IplImage* img;

   /// The list of keypoints
   std::vector<IPoint>& ipts;

   /// Index of current keypoint in above list
   int index;

public:
   /// Standard Constructor (img is an integral image)
   SurfDescriptor(IplImage* img, std::vector<IPoint>& ipts);

   /// Describe all features in the supplied vector
   void getDescriptors(bool rotation_invariant = true);

private:
   /// Assign the current keypoint an orientation
   void getOrientation();

   /// Get the descriptor. See Agrawal ECCV 08
   void getDescriptor(bool rotation_invariant = true);

   /// Calculate the value of the 2d gaussian at x,y
   inline float gaussian(float x, float y, float sig);

   /// Calculate Haar wavelet responses in x and y directions
   inline float haarX(int row, int column, int size);
   inline float haarY(int row, int column, int size);

   /// Get the angle from the +ve x-axis of the vector given by [X Y]
   float getAngle(float X, float Y);

   /// Handy constant to have around
   static const float pi = 3.14159f;
} ;

//---------------------- INLINE MEMBER FUNCTIONS ------------------------

// Calculate the value of the 2d gaussian at x,y
inline float SurfDescriptor::gaussian(float x, float y, float sig)
{
  return 1/(2*pi*sig*sig) * expf(-(x*x + y*y)/(2*sig*sig));
}

// Calculate Haar wavelet responses in x direction
inline float SurfDescriptor::haarX(int row, int column, int s)
{
  return BoxIntegral(img, row-s/2, column,     s, s/2)
       - BoxIntegral(img, row-s/2, column-s/2, s, s/2);
}

// Calculate Haar wavelet responses in y direction
inline float SurfDescriptor::haarY(int row, int column, int s)
{
  return BoxIntegral(img, row, column-s/2,     s/2, s)
       - BoxIntegral(img, row-s/2, column-s/2, s/2, s);
}

//-----------------------------------------------------------------------

} // end of namespace encapsulating this file

#endif
