/***********************************************************
*  --- OpenSURF ---                                        *
*  This library is distributed under the GNU GPL. Please   *
*  contact chris.evans@irisys.co.uk for more information.  *
*                                                          *
*  C. Evans, Research Into Robust Visual Features,         *
*  MSc University of Bristol, 2008.                        *
*                                                          *
************************************************************/

/**
   NOTE: This is a hacked version of Chris Evans's OpenSURF library. It
   was hacked by Manu Viswanathan for use at USC's iLab. Please do not
   bother Chris Evans with problems in this hacked version.

   Some of the changes from the original OpenSURF code base include:
      - file name changes
      - wrapping of all classes and functions in a namespace
      - autotools build system and "librarification" for GNU/Linux
      - class/function name changes
      - minor changes to class/function semantics and external API
      - code clean-up to eliminate compiler warnings
      - etc.

   The actual guts of the implementation/internals performing the SURF
   computations remains the same.

   Nonetheless, do not trust this code to actually do what it purports to
   do. Neither Manu Viswanathan nor iLab assume any responsibility or
   liability of any sort whatsoever.
*/

#ifndef OPENSURF_OPENSURF_DOT_HH
#define OPENSURF_OPENSURF_DOT_HH

//------------------------------ HEADERS --------------------------------

// OpenSURF headers
#include "IPoint.hh"
#include "defaults.hh"

// Standard C++ headers
#include <vector>

//----------------------- FORWARD DECLARATIONS --------------------------

// This should be available from the OpenCV headers
//struct _IplImage ;
typedef struct _IplImage IplImage ;

//----------------------------- NAMESPACE -------------------------------

/**
   \namespace opensurf
   \brief Namespace encapsulating the OpenSURF library.

   All the OpenSURF classes, functions and other definitions are wrapped
   inside of the opensurf namespace.

   \remark This namespace encapsulation is specific to Manu Viswanathan's
   hacked version of the OpenSURF library and is not part of the
   original.
*/
namespace opensurf {

//--------------------------- EXTERNAL API ------------------------------

/**
   \fn std::vector<opensurf::IPoint>
       opensurf::doSurf(IplImage* input_image,
                        bool  rot_invariant,
                        int   octaves, int intervals,
                        int   init_sample,
                        float blob_threshold)

   \brief Returns an std::vector of described interest points.

   \param input_image the input IplImage for which we want SURF key
      points and descriptors
   \param rot_invariant a boolean flag indicating whether the SURF
      algorithm should run in rotation invariant mode or not; defaults to
      true, i.e., by default, the SURF algorithm will run in rotation
      invariant mode; if passed a value of false for this parameter, the
      algorithm will only consider the upright orientation
   \param octaves number of octaves to calculate; default is 4
   \param intervals number of intervals per octave; default is 4
   \param init_sample the initial sampling step; default is 2
   \param blob_threshold the blob response threshold; default is 0.0004

   \return std::vector<opensurf::IPoint>

   This function is the main external interface presented by (the hacked
   version) of the OpenSURF library. Given an input image and several
   parameters to tweak the operation of the SURF algorithm, it computes
   the interest points (aka keypoints) in the input image and then their
   corresponding SURF descriptors. The results are returned in a vector
   of opensurf::IPoint objects. Clients can use the returned IPoint
   objects to query their coordinates as well as their SURF descriptors.

   \remarks This function is specific to this hacked version of the
   OpenSURF library. In the original, it was called surfDetDes() and,
   instead of returning the vector of IPoints via the function return
   value, it did so using a reference to such a beast created by the
   caller. Otherwise, the function is pretty much the same as the
   original.
*/
std::vector<IPoint>
doSurf(IplImage* input_image,
       bool  rot_invariant  = true,
       int   octaves        = DEFAULT_NUM_OCTAVES,
       int   intervals      = DEFAULT_NUM_INTERVALS,
       int   init_sample    = DEFAULT_INITIAL_SAMPLING_STEP,
       float blob_threshold = DEFAULT_BLOB_RESPONSE_THRESHOLD) ;

/**
   \fn std::vector<opensurf::IPoint>
       opensurf::computeSurfKeypoints(IplImage* input_image,
                                      int   octaves,
                                      int   intervals,
                                      int   init_sample,
                                      float blob_threshold)

   \brief Returns an std::vector of SURF keypoints. These keypoints do
   not have their associated SURF descriptors yet.

   \param input_image the input IplImage for which we want SURF keypoints
   \param octaves number of octaves to calculate; default is 4
   \param intervals number of intervals per octave; default is 4
   \param init_sample the initial sampling step; default is 2
   \param blob_threshold the blob response threshold; default is 0.0004

   \return std::vector<opensurf::IPoint>

   This function can be used to compute the SURF interest points (aka
   keypoints) for an input image. The SURF descriptors for these
   keypoints are not computed.

   The function accepts several parameters that can be used to tweak the
   operation of the SURF algorithm.

   The results are returned in a vector of opensurf::IPoint objects.
   Clients can use the returned IPoint objects to query the coordinates
   of the keypoints.

   \remarks This function is specific to this hacked version of the
   OpenSURF library. In the original, it was called surfDet() and,
   instead of returning the vector of IPoints via the function return
   value, it did so using a reference to such a beast created by the
   caller. Otherwise, the function is pretty much the same as the
   original.
*/
std::vector<IPoint>
computeSurfKeypoints(IplImage* input_image,
                     int   octaves        = DEFAULT_NUM_OCTAVES,
                     int   intervals      = DEFAULT_NUM_INTERVALS,
                     int   init_sample    = DEFAULT_INITIAL_SAMPLING_STEP,
                     float blob_threshold = DEFAULT_BLOB_RESPONSE_THRESHOLD) ;

/**
   \fn void
       opensurf::computeSurfDescriptors(IplImage* input_image,
                                        std::vector<opensurf::IPoint>& keypts,
                                        bool rot_invariant)

   \brief Computes the SURF descriptors given the input image and the
   "raw" keypoints.

   \param[in] input_image the input IplImage for which we want SURF
      keypoints' descriptors
   \param[in,out] keypts the std::vector of SURF keypoints computed
      previously
   \param[in] rot_invariant a boolean flag indicating whether the SURF
      algorithm should run in rotation invariant mode or not; defaults to
      true, i.e., by default, the SURF algorithm will run in rotation
      invariant mode; if passed a value of false for this parameter, the
      algorithm will only consider the upright orientation

   This function can be used to compute the SURF descriptors given an
   input image and the SURF keypoints.

   \remarks This function is specific to this hacked version of the
   OpenSURF library. In the original, it was called surfDes(). Apart from
   the name change, the function is pretty much the same as the original.
*/
void
computeSurfDescriptors(IplImage* input_image,
                       std::vector<IPoint>& keypts,
                       bool rot_invariant = true) ;

//-----------------------------------------------------------------------

} // end of namespace encapsulating this file

#endif
