/*!
  \file
  \brief URG Contorl

  \author Satofumi KAMIMURA

  $Id$
*/

#include <stdio.h>
#include <string.h>
#include <math.h>

#include <urg/urg_ctrl.h>
#include <urg/scip_handler.h>
#include <urg/urg_errno.h>
#include <urg/serial_ctrl.h>
#include <urg/serial_utils.h>
#include <urg/serial_errno.h>

#if defined(WINDOWS_OS)
#define snprintf _snprintf
#endif

enum {
  ScipTimeout = 1000,           /*!< [msec] */
  EachTimeout = 10,             /*!< [msec] */
};


/* Connection */
static int urg_firstConnection(urg_t *urg, long baudrate) {

  long try_baudrates[] = { 115200, 19200, 38400 };
  int try_size = sizeof(try_baudrates) / sizeof(try_baudrates[0]);
  int reply = 0;
  int ret;
  int i;

  /* Replace the first one in the array with the wanted Baudrate to be connected */
  for (i = 1; i < try_size; ++i) {
    if (baudrate == try_baudrates[i]) {
      long swap_tmp = try_baudrates[i];
      try_baudrates[i] = try_baudrates[0];
      try_baudrates[0] = swap_tmp;
      break;
    }
  }

  /* Try to connect with the specified baudrate , and check for response */
  for (i = 0; i < try_size; ++i) {

    /* Change host side baudrate */
    ret = serial_setBaudrate(&urg->serial_, try_baudrates[i]);
    if (ret < 0) {
      /* !!! What to do with error message is not yet decided */
      /* error_message_ = con_->what(); */
      return ret;
    }

    /* Issue QT command */
    ret = scip_qt(&urg->serial_, &reply, ScipWaitReply);
    if ((ret < 0) && (reply != 0xE)) {
      /* If MD/MS command is recieved, probably it will enter into this branch */
      /* Read out all the contents received data, then process the next one */
      /* In case of SCIP1.1, (reply == -14) means 'E'.*/

      serial_skip(&urg->serial_, ScipTimeout, EachTimeout);
      reply = 0x00;
    }

    /* If the response is returned, it is already in SCIP2.0 mode and no need to send SCIP2.0 */
    if (reply != 0x00) {

      if (scip_scip20(&urg->serial_) < 0) {
        /* If there is no response , continue with other baudrate */
        continue;
      }
    }
    if (baudrate == try_baudrates[i]) {
      return 0;
    }

    /* Change URG side to specified baudrate */
    if (! scip_ss(&urg->serial_, baudrate)) {
      /* !!! What to do with error message is not yet decided */
      /* error_message_ = scip_->what(); */
      return UrgSsFail;
    } else {
      return 0;
    }
  }

  /* !!! errno should be renewed */
  return UrgAdjustBaudrateFail;
}


/* Open serial device and initialize URG */
int urg_connect(urg_t *urg, const char *device, long baudrate) {

  /* Open serial connection */
  int ret = serial_connect(&urg->serial_, device, baudrate);
  if (ret != 0) {
    urg->errno_ = ret;
    return ret;
  }

  /* Process URG connection  */
  ret = urg_firstConnection(urg, baudrate);
  if (ret < 0) {
    urg->errno_ = ret;
    serial_disconnect(&urg->serial_);
    return ret;
  }

  /* Renew and initialize the parameter information */
  ret = scip_pp(&urg->serial_, &urg->parameters_);
  if (ret < 0) {
    urg->errno_ = ret;
    serial_disconnect(&urg->serial_);
    return ret;
  }
  urg->skip_lines_ = 0;
  urg->skip_frames_ = 0;
  urg->capture_times_ = 0;
  urg->is_laser_on_ = UrgLaserUnknown;

  urg->errno_ = UrgNoError;
  return 0;
}


void urg_disconnect(urg_t *urg) {

  /*  To stop MD/MS command */
  urg_laserOff(urg);

  /* Disconnect serial connection */
  serial_disconnect(&urg->serial_);
}


int urg_isConnected(urg_t *urg) {

  /* Return 0, if serial connectionis valid */
  return serial_isConnected(&urg->serial_);
}


const char *urg_getError(urg_t *urg) {

  return urg_strerror(urg->errno_);
}


int urg_getVersionLines(urg_t *urg, char* lines[], int lines_max) {

  return scip_vv(&urg->serial_, lines, lines_max);
}


/* Send PP command. Analyse and store the response , then return*/
int urg_getParameters(urg_t *urg, urg_parameter_t* parameters) {

  int ret = 0;

  ret = scip_pp(&urg->serial_, &urg->parameters_);
  if (parameters) {
    *parameters = urg->parameters_;
  }

  urg->errno_ = UrgNoError;
  return 0;
}


int urg_getDataMax(urg_t *urg) {

  return urg->parameters_.area_max_ + 1;
}


int urg_getScanMsec(urg_t *urg) {

  int scan_rpm = urg->parameters_.scan_rpm_;
  return (scan_rpm <= 0) ? 1 : (1000 * 60 / scan_rpm);
}


long urg_getDistanceMax(urg_t *urg) {

  return urg->parameters_.distance_max_;
}


long urg_getDistanceMin(urg_t *urg) {

  return urg->parameters_.distance_min_;
}


int urg_setSkipLines(urg_t *urg, int lines) {

  /*  Register number of skiped lines */
  urg->skip_lines_ = 0;

  return 0;
}


int urg_setSkipFrames(urg_t *urg, int frames) {

  /*  Register number of skip frames */
  urg->skip_frames_ = frames;

  return 0;
}


int urg_setCaptureTimes(urg_t *urg, int times) {

  /* Register frequency at which MD/MS data is received */
  urg->capture_times_ = times;

  return 0;
}


int urg_requestData(urg_t *urg,
                    urg_request_type request_type,
                    int first_index,
                    int last_index) {

  char buffer[] = "MDsssseeeellstt\r";

  if ((first_index == URG_FIRST) && (last_index == URG_LAST)) {
    first_index = 0;
    last_index = urg->parameters_.area_max_;
  }

  if ((request_type == URG_GD) || (request_type == URG_GS)) {
    /* In case of GD/GS command  */
    snprintf(buffer, 14, "G%c%04d%04d%02d\r",
             ((request_type == URG_GD) ? 'D' : 'S'),
             first_index, last_index,
             urg->skip_lines_);

    /* If laser is not ON, switch ON the laser*/
    if (urg->is_laser_on_ != UrgLaserOn) {
      urg_laserOn(urg);
    }

  } else if ((request_type == URG_MD) || (request_type == URG_MS)) {
    /* In case of MD / MS command */
    snprintf(buffer, 17, "M%c%04d%04d%02d%d%02d\r",
             ((request_type == URG_MD) ? 'D' : 'S'),
             first_index, last_index,
             urg->skip_lines_,
             urg->skip_frames_,
             urg->capture_times_);

  } else {
    return UrgInvalidArgs;
  }

  return scip_send(&urg->serial_, buffer);
}


/* Decode URG's 6 bit data */
static long decode(const char* data, int data_byte) {

  long value = 0;
  int i;

  for (i = 0; i < data_byte; ++i) {
    value <<= 6;
    value &= ~0x3f;
    value |= data[i] - 0x30;
  }
  return value;
}


static int convertRawData(long data[],
                          const char* buffer, int buffer_size,int filled,
                          int data_bytes) {

  static char remain_data[3];
  static int remain_byte = 0;

  int n;
  int i;

  if (filled == 0) {
    /* Initialize the number of data, which are remained when the first time is called */
    remain_byte = 0;
  }

  /* If there is any data left, then process that data */
  if (remain_byte != 0) {
    memcpy(&remain_data[remain_byte], buffer, data_bytes - remain_byte);
    data[filled++] = decode(remain_data, data_bytes);
  }

  /* Process one line of data */
  n = buffer_size - data_bytes;
  for (i = (data_bytes - remain_byte) % data_bytes; i < n; i += data_bytes) {
    data[filled++] = decode(&buffer[i], data_bytes);
  }

  /* Save the remaining data */
  remain_byte = buffer_size - i;
  memcpy(remain_data, &buffer[i], remain_byte);

  return filled;
}


/* Receive the data */
int urg_receiveData(urg_t *urg, long data[], int data_max) {

  enum {
    EchoBack = 0,
    ReplyCode,
    Timestamp,

    False = 0,
    True = 1,

    MD_MS_Length = 15,          /*!< Length of MD, MS command  */
    GD_GS_Length = 12,          /*!< Length of GD, GS command */
  };

  int lines = 0;
  char buffer[UrgLineWidth];
  int filled = 0;
  int is_header = False;
  int n;

  char current_type[] = "xx";
  int current_first = -1;
  int current_last = -1;
  int current_skip_lines = -1;
  int current_skip_frames = -1;
  int current_capture_times = -1;
  int current_data_bytes = 3;

  /* Initialization of time stamp */
  urg->last_timestamp_ = UrgInvalidTimestamp;

  while (1) {
    n = serial_getLine(&urg->serial_, buffer, ScipLineWidth, ScipTimeout);
    if (n <= 0) {
      if (is_header) {
        /* !!! Have to do something to this strange structure */
        is_header = False;
        lines = 0;
        continue;
      }
      break;
    }

    /* fprintf(stderr, "[%s], %d\n", buffer, n); */

    switch (lines) {
    case EchoBack:
      /* Echo back */

      if ((n != GD_GS_Length) && (n != MD_MS_Length)) {
        /* Return if response is not GD/GS, MD/MS */
        return -1;
      }
      /* Response command */
      current_type[0] = buffer[0];
      current_type[1] = buffer[1];

      /* Initialisation of receiving settings */
      current_first = -1;
      current_last = -1;
      current_skip_lines = -1;

      if (n == GD_GS_Length) {
        /* Ignore receive frame settings and number of frames settings for GD/GS command */
        current_skip_frames = -1;
        current_capture_times = -1;
      }
      current_data_bytes = (current_type[1] == 'S') ? 2 : 3;
      break;

    case ReplyCode:
      /* Response */
      /* !!! If response is not equal to 00, then it is an error and hence return */
      /* !!! */

      /* In case of MD/MS, response = "00" means transition requenst and hence */
      /* readout one more line, and then reset the process */
      if (current_type[0] == 'M' &&
          (! strncmp(buffer, "00", 2))) {
        is_header = True;
      }

      /* !!! "99b" is actual data */
      /* if (! strcmp(buffer, "99b")) { */
      /* } */
      break;

    case Timestamp:
      /* Time stamp */
      urg->last_timestamp_ = decode(buffer, 4);
      break;

    default:
      /* Data conversion process */
      filled = convertRawData(data, buffer, n - 1, filled, current_data_bytes);

      break;
    }
    ++lines;
  }

  /* !!! Receive requested data */
  /* !!! Store the latest data information in the URG.   */

  return filled;
}


int urg_receiveRangedData(urg_t *urg, long data[], int data_max,
                          int first_index, int last_index) {

  /* !!!  Receive part of data*/
  /* Not implemented */

  /* !!! */
  return -1;
}


long urg_getRecentTimestamp(urg_t *urg) {

  /* Return last time stamp */
  return urg->last_timestamp_;
}


double urg_index2rad(urg_t *urg, int index) {

  double radian = (2.0 * M_PI) *
    (index - urg->parameters_.area_front_) / urg->parameters_.area_total_;

  return radian;
}


int urg_index2deg(urg_t *urg, int index) {

  int degree = (int)(urg_index2rad(urg, index) * 180 / M_PI);

  return degree;
}


int urg_rad2index(urg_t *urg, double radian) {

  int index =
    (int)((radian * urg->parameters_.area_total_) / (2.0*M_PI))
    + urg->parameters_.area_front_;

  if (index < 0) {
    index = 0;
  } else if (index > urg->parameters_.area_max_) {
    index = urg->parameters_.area_max_;
  }
  return index;
}


int urg_deg2index(urg_t *urg, int degree) {

  return urg_rad2index(urg, M_PI * degree / 180.0);
}


int urg_laserOn(urg_t *urg) {

  /* Send BM command */
  int expected_ret[] = { 0, 2, -1 };
  int send_n = scip_send(&urg->serial_, "BM\r");
  if (send_n != 3) {
    /* !!! urg->errno = UrgSendFail; */
    return SerialSendFail;
  }
  if (scip_recv(&urg->serial_, NULL, expected_ret, ScipTimeout) == 0) {
    urg->is_laser_on_ = UrgLaserOn;
  }

  return 0;
}


int urg_laserOff(urg_t *urg) {

  /* Send QT command */
  return scip_qt(&urg->serial_, NULL, ScipWaitReply);
}


int urg_enableTimestampMode(urg_t *urg) {

  /* Send TM0 command */
  int expected_ret[] = { 0, 2, -1 };
  int send_n = scip_send(&urg->serial_, "TM0\r");
  if (send_n != 4) {
    /* !!! urg->errno = UrgSendFail; */
    return SerialSendFail;
  }
  return scip_recv(&urg->serial_, NULL, expected_ret, ScipTimeout);
}


int urg_disableTimestampMode(urg_t *urg) {

  /* Send TM2 command */
  int expected_ret[] = { 0, 3, -1 };
  int send_n = scip_send(&urg->serial_, "TM2\r");
  if (send_n != 4) {
    /* !!! urg->errno = UrgSendFail; */
    return SerialSendFail;
  }
  return scip_recv(&urg->serial_, NULL, expected_ret, ScipTimeout);
}


long urg_getCurrentTimestamp(urg_t *urg) {

  char buffer[ScipLineWidth];
  long timestamp = -1;
  int ret = 0;
  int n;

  /* Send TM1 command*/
  int expected_ret[] = { 0, -1 };
  int send_n = scip_send(&urg->serial_, "TM1\r");
  if (send_n != 4) {
    /* !!! urg->errno = UrgSendFail; */
    return SerialSendFail;
  }
  ret = scip_recv(&urg->serial_, NULL, expected_ret, ScipTimeout);
  if (ret != 0) {
    return ret;
  }

  // Return the decoded time stamp
  n = serial_getLine(&urg->serial_, buffer, ScipLineWidth, ScipTimeout);
  if (n == 5) {
    timestamp = decode(buffer, 4);
  }

  // Last character of the reply is read but not used.
  n = serial_recv(&urg->serial_, buffer, 1, ScipTimeout);
  if (! serial_isLF(buffer[0])) {
    serial_ungetc(&urg->serial_, buffer[0]);
  }

  return timestamp;
}


int urg_setMotorSpeed(urg_t *urg, int speed_percent) {

  /* Not implemented */

  return -1;
}
