#ifndef C_URG_CTRL_H
#define C_URG_CTRL_H

/*!
  \file
  \brief URG Control

  \author Satofumi KAMIMURA

  $Id$

*/

#include <urg/urg_t.h>

#ifdef __cplusplus
extern "C" {
#endif

enum {
  UrgLineWidth = 64 + 1 + 1,    /*!< Maximum line width */
  UrgInfinityTimes = 0,         /*!<  Data receive count during MD/MS command */
};


/*!
  \brief Command type of URG
*/
typedef enum {
  URG_GD,                       /*!< GD Command */
  URG_GS,                       /*!< GS Command */
  URG_MD,                       /*!< MD Command */
  URG_MS,                       /*!< MS Command */
} urg_request_type;


/*!
  \brief Parameters to skip specified range of URG data.
*/
enum {
  URG_FIRST = -1,		   /*!<  Start position ,while receiving complete data  */
  URG_LAST = -1,		   /*!<  End position, while receiving complete data */

  UrgInvalidTimestamp = -1,		/*!< Receive data for infinite times */
};


/*!
  \brief Connection

  \param urg [i/o] Structure of URG control
  \param device [i] Connected device
  \param baudrate [i] Connected baudrate

  \retval 0 Normal
  \retval < 0 Error

  Example use
  \code 
urg_t urg;

// Connection
if (urg_connect(&urg, "/dev/ttyACM0", 115200) < 0) {
  printf("urg_connect: %s\n", urg_getError(&urg));
  return -1;
}

...

urg_disconnect(&urg); \endcode
*/
extern int urg_connect(urg_t *urg, const char *device, long baudrate);


/*!
  \brief Disconnect

  \param urg [i/o] Structure of URG control

  \see urg_connect
*/
extern void urg_disconnect(urg_t *urg);


/*!
  \brief Check wheather connected or not

  \param urg [i/o] Structure of URG control

  \retval 0 if connected
  \retval < 0 if not connected

  Example use
  \code
  if (urg_isConnected(&urg) == 0) {
    printf("connected.\n");
  } else {
    printf("not connected.\n");
  } \endcode

  \see urg_connect, urg_disconnect
*/
extern int urg_isConnected(urg_t *urg);


/*!
  \brief  Get error message

 Returns the error message. How ever not in all cases.

  \param urg [i/o] Structure of URG control 

  \return  Error message

  \see urg_connect
*/
extern const char *urg_getError(urg_t *urg);


/*!
  \brief Get version details

  \param urg [i/o] Structure of URG control
  \param lines [o] Buffer used to store version details
  \param lines_max Maximum number of lines 

  \retval 0 Normal
  \retval < 0 Error

  \attention Maximum number of characters in a line of buffer should not exceeed 65 bytes

  \see get_version_lines.c
*/
extern int urg_getVersionLines(urg_t *urg, char* lines[], int lines_max);


/*!
  \brief  Return URG parameter

  \param urg [i/o] Structure of URG control
  \param parameters [o] Structure of URG parameter.

  \retval 0 Normal
  \retval < 0 Success

  \see get_parameters.c
 */
extern int urg_getParameters(urg_t *urg, urg_parameter_t* parameters);


/*!
  \brief  Returns number of maximum data in one scan 

  \param urg [i/o]  Structure of URG control

  \retval >= 0  Number of maximum data in one scan
  \retval < 0 Error

  \see gd_scan.c
*/
extern int urg_getDataMax(urg_t *urg);


/*!
  \brief Returns the time taken to complete one scan

  \param urg [i/o] Structure of URG contorl

  \retval >= 0 Time taken to complete one scan [msec]  
  \retval < 0 Error

  \attention Returns measurement time when the motor speed is 100% matches with specified speed

  \see urg_setMotorSpeed

  \see md_scan.c
*/
extern int urg_getScanMsec(urg_t *urg);


/*!
  \brief Maximum measurable distance

  \param urg [i/o] Structure of URG control

  \retval >= 0 Maximum measurable distance [mm]
  \retval < 0 Error

  \see expand_2d.c
*/
extern long urg_getDistanceMax(urg_t *urg);


/*!
  \brief Minimum measurable distance

  \param urg [i/o] Structure of URG control

  \retval >= 0 Minimum measurable distance [mm]
  \retval < 0 Error

  \see expand_2d.c
*/
extern long urg_getDistanceMin(urg_t *urg);


//////////////////////////////////////////////////////////////////////
/*!
  \brief Set the number of adjacent steps that can be merged into single data

  Quantity of receive can be reduced.

  \param urg [i/o] Structure of URG control
  \param lines [i] number of adjacent steps that can be merged into single data

  \retval 0 Normal
  \retval < Error
*/
extern int urg_setSkipLines(urg_t *urg, int lines);


/*!
  \brief  Skipping the number of scans when obtaining multiple scan data can be set here


  \param urg [i/o] URG Structure of URG control
  \param frames [i] Number of scans that can be skipped

  \retval 0 Normal
  \retval < 0 Error

  \attention This applies only for MD/MS command.
*/
extern int urg_setSkipFrames(urg_t *urg, int frames);


/*!
  \brief Number of continous scan data count can be set here.

  \param urg [i/o] URG Structure of URG control
  \param times [i] Number of scan data.

  \retval 0 Normal
  \retval < 0 Error

  \attention This applies only for MD/MS command.
  \attention To receive data continuously for more than 100 times, please specify UrgInfinityTimes.

  \code
//  To receive data infinitely.
urg_setCaptureTimes(&urg, UrgInfinityTimes);

...

// If laser is switched off, then data receive will be stoped.
urg_laser_off(&urg);
  \endcode
*/
extern int urg_setCaptureTimes(urg_t *urg, int times);


/*!
  \brief  Request for distance data

  Request for the distance data between [first_index, last_index].By assigning the URG_FIRST, URG_LAST to first_index and last_index respectively, complete range of 
  data can be received. 

  \param urg [i/o] Structure of URG control
  \param request_type [i] Data request type
  \param first_index [i]  Data receive begin index
  \param last_index [i]   Data receieve end index

  \retval 0 Normal
  \retval < 0 Error

  \see gd_scan.c, md_scan.c
*/
extern int urg_requestData(urg_t *urg,
                           urg_request_type request_type,
                           int first_index,
                           int last_index);


/*!
  \brief Receive URG data

  \param urg [i/o] Structure of URG control
  \param data [o]  Storage location of received data
  \param data_max [i]  Maximum number of data that can be received 

  \retval 0 > Number of received data
  \retval < 0 Error
*/
extern int urg_receiveData(urg_t *urg, long data[], int data_max);


/*!
  \brief  Receive part of URG data

  \param urg [i/o]  Structure of URG control
  \param data [o]  Storage location of received data
  \param data_max [i] Maximum number of data that can be received 
  \param first_index [i] Data receive begin index
  \param last_index [i]  Data receieve end index

  \retval 0 > Number of received data
  \retval < 0 Error

  \attention Not implemented

  \see gd_scan.c, md_scan.c
*/
extern int urg_receiveRangedData(urg_t *urg, long data[], int data_max,
                                 int first_index, int last_index);


/*!
  \brief  Receive time stamp

  \param urg [i/o] Structure of URG control

  \retval Time stamp [msec]

  \see md_scan.c
*/
extern long urg_getRecentTimestamp(urg_t *urg);


//////////////////////////////////////////////////////////////////////


/*!
  \brief  Change index value into angle

  \param urg [i/o] Structure of URG control
  \param index [i] Index value

  \retval 0 Normal
  \retval < 0 Error

  \attention Returns the latest converted value.

  \see index_convert.c
*/
extern double urg_index2rad(urg_t *urg, int index);


/*!
  \brief Change index value into angle

  \param urg [i/o] Structure of URG control
  \param index [i] Index value

  \retval 0 Normal
  \retval < 0 Error

  \attention Returns the latest converted value.

  \see index_convert.c
*/
extern int urg_index2deg(urg_t *urg, int index);


/*!
  \brief Convert from angle to index value.

  \param urg [i/o] Structure of URG control
  \param radian [i] Angle

  \retval 0 Normal
  \retval < 0 Error

  \see index_convert.c
*/
extern int urg_rad2index(urg_t *urg, double radian);


/*!
  \brief Convert from angle to index value.

  \param urg [i/o] Structure of URG control
  \param degree [i] Angle

  \retval 0 Normal
  \retval < 0 Error

  \see index_convert.c
*/
extern int urg_deg2index(urg_t *urg, int degree);


//////////////////////////////////////////////////////////////////////


/*!
  \brief Instructs laser to ON

  \param urg [i/o] Structure of URG control

  \retval 0 Normal
  \retval < 0 Error

  \see gd_scan.c
*/
extern int urg_laserOn(urg_t *urg);


/*!
  \brief Instructs laser to OFF

  \param urg [i/o] Structure of URG control

  \retval 0 Normal
  \retval < 0 Error
*/
extern int urg_laserOff(urg_t *urg);


//////////////////////////////////////////////////////////////////////


/*!
  \brief Enter into time stamp mode

  \param urg [i/o] Structure of URG control

  \retval 0 Normal
  \retval < 0 Error
*/
extern int urg_enableTimestampMode(urg_t *urg);


/*!
  \brief Exit from time stamp mode

  \param urg [i/o] Structure of URG control

  \retval 0 Normal
  \retval < 0 Error
*/
extern int urg_disableTimestampMode(urg_t *urg);


/*!
  \brief Get time stamp

  \param urg [i/o] Structure of URG control

  \retval >= 0 Time stamp [msec]
  \retval < 0 Error
*/
extern long urg_getCurrentTimestamp(urg_t *urg);


//////////////////////////////////////////////////////////////////////

/*!
  \brief Instructs to adjust the sensor’s motor speed.

  \param urg [i/o] URG Structure of URG control
  \param speed_percent [i] percent Number of revolution of motor [%]

  \attention Not implemented
*/
extern int urg_setMotorSpeed(urg_t *urg, int speed_percent);

#ifdef __cplusplus
}
#endif

#endif /* !C_URG_CTRL_H */
