/***********************************************************
*  --- OpenSURF ---                                        *
*  This library is distributed under the GNU GPL. Please   *
*  contact chris.evans@irisys.co.uk for more information.  *
*                                                          *
*  C. Evans, Research Into Robust Visual Features,         *
*  MSc University of Bristol, 2008.                        *
*                                                          *
************************************************************/

/**
   NOTE: This is a hacked version of Chris Evans's OpenSURF library. It
   was hacked by Manu Viswanathan for use at USC's iLab. Please do not
   bother Chris Evans with problems in this hacked version.

   Some of the changes from the original OpenSURF code base include:
      - file name changes
      - wrapping of all classes and functions in a namespace
      - autotools build system and "librarification" for GNU/Linux
      - class/function name changes
      - minor changes to class/function semantics and external API
      - code clean-up to eliminate compiler warnings
      - etc.

   The actual guts of the implementation/internals performing the SURF
   computations remains the same.

   Nonetheless, do not trust this code to actually do what it purports to
   do. Neither Manu Viswanathan nor iLab assume any responsibility or
   liability of any sort whatsoever.
*/

#ifndef OPENSURF_FAST_HESSIAN_DOT_HH
#define OPENSURF_FAST_HESSIAN_DOT_HH

//------------------------------ HEADERS --------------------------------

// OpenSURF headers
#include "IPoint.hh"

// Standard C++ headers
#include <vector>

// Standard C headers
#include <math.h>

//----------------------- FORWARD DECLARATIONS --------------------------

// These should be available from the OpenCV headers
typedef struct _IplImage IplImage ;
typedef struct CvMat CvMat ;

//----------------------------- NAMESPACE -------------------------------

/**
   \namespace opensurf
   \brief Namespace encapsulating the OpenSURF library.

   All the OpenSURF classes, functions and other definitions are wrapped
   inside of the opensurf namespace.

   \remark This namespace encapsulation is specific to Manu Viswanathan's
   hacked version of the OpenSURF library and is not part of the
   original.
*/
namespace opensurf {

//------------------------- CLASS DEFINITION ----------------------------

class FastHessian {
   /// The fast Hessian operates on an integral image. These data members
   /// are used to keep track of it.
   //@{
   IplImage* img ;
   int i_width, i_height ;
   //@}

   /// Once it's done, the fast Hessian operation will return a vector of
   /// keypoints. This vector must be created by the client and its
   /// reference passed to this object.
   std::vector<IPoint>& ipts ;

   /// Number of octaves.
   int octaves ;

   /// Number of intervals per octave.
   int intervals ;

   /// Initial sampling step for keypoint detection
   int init_sample ;

   /// Threshold value for blob responses.
   float thres ;

   /// Array stack of determinant of Hessian values
   float* m_det ;

public:
   /// Constructor without image.
   FastHessian(std::vector<IPoint>& ipts,
               const int octaves,     const int   intervals,
               const int init_sample, const float blob_threshold) ;

   /// Constructor with image.
   FastHessian(IplImage* img,
               std::vector<IPoint>& ipts,
               const int octaves,     const int   intervals,
               const int init_sample, const float blob_threshold) ;

   /// Save the parameters.
   void saveParameters(const int octaves,
                       const int intervals,
                       const int init_sample,
                       const float blob_threshold) ;

   /// Set or re-set the integral image source.
   void setIntegralImage(IplImage* img) ;

   /// Find the image features and write into vector of features
   void getIPoints() ;

private:
   /// Calculate determinant of hessian responses.
   void buildDet() ;

   /// Non-maximal suppression function.
   int isExtremum(int octave, int interval, int column, int row) ;

   /// Return the value of the approximated determinant of Hessian.
   inline float getVal(int octave, int interval, int col, int row) ;

   /// Return the sign of the laplacian (trace of the Hessian)
   inline int getLaplacian(int octave, int interval, int col, int row) ;

   /// Interpolation functions - adapted from Lowe's SIFT implementation.
   //@{
   void interpolateExtremum(int octv, int intvl, int r, int c) ;
   void interpolateStep( int octv, int intvl, int r, int c,
                         double* xi, double* xr, double* xc) ;
   CvMat* deriv3D( int octv, int intvl, int r, int c) ;
   CvMat* hessian3D(int octv, int intvl, int r, int c) ;
   //@}

public:
   /// Clean-up
   ~FastHessian();
} ;

//---------------------- INLINE MEMBER FUNCTIONS ------------------------

// Return the value of the approximated determinant of Hessian
inline float FastHessian::getVal(int o, int i, int c, int r)
{
   return fabs(m_det[  (o * intervals + i) * (i_width * i_height)
                     + (r * i_width + c)]) ;
}

// Return the sign of the laplacian (trace of the Hessian)
inline int FastHessian::getLaplacian(int o, int i, int c, int r)
{
  float res = (m_det[  (o * intervals + i) * (i_width * i_height)
                     + (r * i_width + c)]) ;

  return (res >= 0) ? 1 : -1;
}

//-----------------------------------------------------------------------

} // end of namespace encapsulating this file

#endif
