/***********************************************************
*  --- OpenSURF ---                                        *
*  This library is distributed under the GNU GPL. Please   *
*  contact chris.evans@irisys.co.uk for more information.  *
*                                                          *
*  C. Evans, Research Into Robust Visual Features,         *
*  MSc University of Bristol, 2008.                        *
*                                                          *
************************************************************/

/**
   NOTE: This is a hacked version of Chris Evans's OpenSURF library. It
   was hacked by Manu Viswanathan for use at USC's iLab. Please do not
   bother Chris Evans with problems in this hacked version.

   Some of the changes from the original OpenSURF code base include:
      - file name changes
      - wrapping of all classes and functions in a namespace
      - autotools build system and "librarification" for GNU/Linux
      - class/function name changes
      - minor changes to class/function semantics and external API
      - code clean-up to eliminate compiler warnings
      - etc.

   The actual guts of the implementation/internals performing the SURF
   computations remains the same.

   Nonetheless, do not trust this code to actually do what it purports to
   do. Neither Manu Viswanathan nor iLab assume any responsibility or
   liability of any sort whatsoever.
*/

//------------------------------ HEADERS --------------------------------

// OpenSURF headers
#include "config.h"
#include "Integral.hh"

// OpenCV headers
#ifdef OPENSURF_HAVE_OPENCV
   #include <opencv/cv.h>
#else
   #error "Sorry, libopensurf needs OpenCV"
#endif

// Standard C++ headers
#include <algorithm>

//----------------------------- NAMESPACE -------------------------------

namespace opensurf {

//------------------------------ HELPERS --------------------------------

// Convert image to single channel 32F
static IplImage* getGray(const IplImage* img)
{
  // Check we have been supplied a non-null img pointer
  if (! img)
     return 0 ;

  IplImage* gray8 ;
  IplImage* gray32;

  gray32 = cvCreateImage(cvGetSize(img), IPL_DEPTH_32F, 1) ;

  if (img->nChannels == 1)
    gray8 = reinterpret_cast<IplImage*>(cvClone(img)) ;
  else
  {
     gray8 = cvCreateImage(cvGetSize(img), IPL_DEPTH_8U, 1) ;
     cvCvtColor(img, gray8, CV_BGR2GRAY) ;
  }

  cvConvertScale(gray8, gray32, 1/255.0, 0) ;

  cvReleaseImage(& gray8) ;
  return gray32;
}

//-------------------------------- API ----------------------------------

// Compute the integral image of input image. Assumes source image to be
// a 32-bit floating point. Returns IplImage in 32-bit float form.
IplImage* Integral(IplImage* input_image)
{
   // Convert the image to single channel 32f
   IplImage* gray_image = getGray(input_image) ;
   IplImage* integral_image
      = cvCreateImage(cvGetSize(gray_image), IPL_DEPTH_32F, 1) ;

   // Set up variables for data access
   int width     = gray_image->width ;
   int height    = gray_image->height ;
   int step      = gray_image->widthStep/sizeof(float) ;
   float* data   = reinterpret_cast<float*>(gray_image->imageData) ;
   float* i_data = reinterpret_cast<float*>(integral_image->imageData) ;

   // First row only
   float rs = 0 ;
   for (int j = 0; j < width; ++j)
   {
      rs += data[j] ;
      i_data[j] = rs ;
   }

   // Remaining cells are sum above and to the left
   for (int i = 1; i < height; ++i)
   {
      rs = 0 ;
      for (int j = 0; j < width; ++j)
      {
         const int k = step * i + j ;
         rs += data[k] ;
         i_data[k] = rs + i_data[k - step] ;
      }
   }

   cvReleaseImage(& gray_image) ;
   return integral_image ; // to be released by client
}

// Compute the sum of pixels within the rectangle specified by the
// top-left start co-ordinate and size.
float BoxIntegral(IplImage* img, int row, int col, int rows, int cols)
{
  float* data = reinterpret_cast<float*>(img->imageData) ;
  int step = img->widthStep/sizeof(float) ;

  // The subtraction by one for row/col is because row/col is inclusive.
  int r1 = std::min(row, img->height) - 1 ;
  int c1 = std::min(col, img->width)  - 1 ;
  int r2 = std::min(row + rows, img->height) - 1 ;
  int c2 = std::min(col + cols, img->width)  - 1 ;

  float A(0), B(0), C(0), D(0) ;
  if (r1 >= 0 && c1 >= 0) A = data[r1 * step + c1] ;
  if (r1 >= 0 && c2 >= 0) B = data[r1 * step + c2] ;
  if (r2 >= 0 && c1 >= 0) C = data[r2 * step + c1] ;
  if (r2 >= 0 && c2 >= 0) D = data[r2 * step + c2] ;

  return std::max(0.0f, A - B - C + D) ;
}

//-----------------------------------------------------------------------

} // end of namespace encapsulating this file
