/*!
  \file
  \brief Process SKIP commands

  \author Satofumi KAMIMURA

  $Id$
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <urg/scip_handler.h>
#include <urg/serial_errno.h>
#include <urg/serial_ctrl.h>
#include <urg/serial_utils.h>
#include <urg/urg_errno.h>

enum {
  /* !!! In common with urg_ctrl.c */
  ScipTimeout = 1000,            /*!< [msec] */
  EachTimeout = 10,             /*!< [msec] */
};

/* !!! static */
/* !!! Decides the checksum of recieved line */
/* !!! Add the argument to discriminate the version information line */
/* !!! Also works as range check function */


/* Send command */
int scip_send(serial_t *serial, const char *send_command) {

  int n = strlen(send_command);
  return serial_send(serial, send_command, n);
}


/*!
   Receive the response from command

  \todo Test the checksum
*/
int scip_recv(serial_t *serial,
              int* return_code, int expected_ret[], int timeout) {

  char recv_ch = '\0';
  int ret_code = 0;
  int n;
  int i;

  /* Receive the response */
  char buffer[ScipLineWidth];

  /*  Skip the first response */
  n = serial_getLine(serial, buffer, ScipLineWidth, timeout);
  if (n <= 0) {
    return SerialRecvFail;
  }

  /* Read the response character array and wait for a while*/
  n = serial_getLine(serial, buffer, ScipLineWidth, timeout);

  /* If the last character is a linefeed character,then read and throw that character */
  n = serial_recv(serial, &recv_ch, 1, timeout);
  if (! serial_isLF(recv_ch)) {
    serial_ungetc(serial, recv_ch);
  }

  /* Return 0 , if the response character is same as anticipated character */
  ret_code = strtol(buffer, NULL, 16);
  if (return_code != NULL) {
    *return_code = ret_code;
  }
  for (i = 0; expected_ret[i] != -1; ++i) {
    if (ret_code == expected_ret[i]) {
      return 0;
    }
  }

  return ret_code;
}


/* Transition to SCIP2.0 */
int scip_scip20(serial_t *serial) {

  int expected_ret[] = { 0x0, 0xE, -1 };
  int ret;

  ret = scip_send(serial, "SCIP2.0\r");
  if (ret != 8) {
    return ret;
  }
  ret = scip_recv(serial, NULL, expected_ret, ScipTimeout);

  return ret;
}


/* Send QT command */
int scip_qt(serial_t *serial, int *return_code, int wait_reply) {

  int expected_ret[] = { 0x0, -1 };
  int ret;

  ret = scip_send(serial, "QT\r");
  if (ret != 3) {
    return ret;
  }

  if (wait_reply == ScipNoWaitReply) {
    return 0;
  }

  ret = scip_recv(serial, return_code, expected_ret, ScipTimeout);
  if (return_code && (*return_code == 0xE)) {
    return UrgScip10;
  }

  return ret;
}


/* Retrieve information from PP command*/
int scip_pp(serial_t *serial, urg_parameter_t *parameters) {

  int send_n;
  int ret = 0;
  int expected_reply[] = { 0x0, -1 };
  int n;
  int i;

  char buffer[ScipLineWidth];

  /* Send PP command */
  send_n = scip_send(serial, "PP\r");
  if (send_n != 3) {
    /* !!! urg->errno = UrgSendFail; */
    return SerialSendFail;
  }

  /* Receive the response */
  ret = scip_recv(serial, NULL, expected_reply, ScipTimeout);
  if (ret < 0) {
    /* urg->errno = UrgRecvFail; */
    return ret;
  }

  /* Receive parameter character array */
  for (i = 0; i < UrgParameterLines; ++i) {
    n = serial_getLine(serial, buffer, ScipLineWidth, ScipTimeout);
    if (n <= 0) {
      return ret;
    }

    /* !!! Validation should be done for the character array like AMIN etc   */

    if (i == 1) {
      parameters->distance_min_ = atoi(&buffer[5]);

    } else if (i == 2) {
      parameters->distance_max_ = atoi(&buffer[5]);

    } else if (i == 3) {
      parameters->area_total_ = atoi(&buffer[5]);

    } else if (i == 4) {
      parameters->area_min_ = atoi(&buffer[5]);

    } else if (i == 5) {
      parameters->area_max_ = atoi(&buffer[5]);

    } else if (i == 6) {
      parameters->area_front_ = atoi(&buffer[5]);

    } else if (i == 7) {
      parameters->scan_rpm_ = atoi(&buffer[5]);
    }
  }

  return 0;
}


/* Receive response from VV command */
int scip_vv(serial_t *serial, char *lines[], int lines_max) {

  int send_n;
  int ret = 0;
  int expected_reply[] = { 0x0, -1 };
  int n;
  int i;

  /* Initialize with an empty message */
  for (i = 0; i < lines_max; ++i) {
    *lines[i] = '\0';
  }

  /* Send VV command */
  send_n = scip_send(serial, "VV\r");
  if (send_n != 3) {
    /* !!! urg->errno = UrgSendFail; */
    return SerialSendFail;
  }

  /* Recieve the response */
  ret = scip_recv(serial, NULL, expected_reply, ScipTimeout);
  if (ret < 0) {
    /* urg->errno = UrgRecvFail; */
    return ret;
  }

  /* Receive version character array */
  for (i = 0; i < lines_max; ++i) {
    n = serial_getLine(serial, lines[i], ScipLineWidth, ScipTimeout);
    if (n <= 0) {
      return ret;
    }
  }

  serial_skip(serial, ScipTimeout, EachTimeout);
  return ret;
}


/* Change according to SS command */
int scip_ss(serial_t *serial, long baudrate) {

  int expected_reply[] = { 0x0, 0x3, -1 };
  int send_n;
  int ret;

  /* !!! Should be error if the requested baudrate is not prescripted baudrate*/

  /* Send SS command */
  char buffer[] = "SSxxxxxx\r";
  snprintf(buffer, 10, "SS%06ld\r", baudrate);
  send_n = scip_send(serial, buffer);
  if (send_n != 9) {
    /* !!! urg->errno = UrgSendFail; */
    return SerialSendFail;
  }

  /* Receive response from command */
  ret = scip_recv(serial, NULL, expected_reply, ScipTimeout);
  if (ret < 0) {
    /* urg->errno = UrgRecvFail; */
    return ret;
  }

  return 0;
}
