#ifndef C_SCIP_HANDLER_H
#define C_SCIP_HANDLER_H

/*!
  \file
  \brief Processing SCIP command

  \author Satofumi KAMIMURA

  $Id$
*/

#include <urg/urg_parameter_t.h>
#include <urg/serial_t.h>

#ifdef __cplusplus
extern "C" {
#endif

enum {
  ScipNoWaitReply = 0,       /*!< Does not wait for SCIP reply */
  ScipWaitReply = 1,          /*!< Wait for SCIP reply */

  ScipLineWidth = 64 + 1 + 1,  /*!< Maximum length of 1 line */
};


/*!
  \brief Send command

  \param serial [o] Struct of serial control
  \param send_command [i] Command to be send

  \retval 0 Success
  \retval < 0 Error
*/
extern int scip_send(serial_t *serial, const char *send_command);


/*!
  \brief Receive command's reply

  If ret is not equal to NULL, then reply of the command will be received and stored in this function. If the reply of command is regarded as normal then in expected_ret terminal can be specified as -1.  If the reply of command consists of expected_ret, then the return value of this function is zero(Normal)\n
  expected_ret exists because in some case other than zero is regarded as normal.

  \param serial [o] Struct of serial control
  \param ret [o] Return value
  \param expected_ret [i] Expected return value
  \param timeout [i] Time out [msec]

  \retval 0 Normal / success
  \retval < 0 Error
*/
extern int scip_recv(serial_t *serial, int* ret, int expected_ret[],
                     int timeout);


/*!
  \brief Transit ot SCIP2.0 mode

  Return zero(Normal) ,if transit to SCIP2.0

  \param serial [i/o] Struct of serial control

  \retval 0 Normal
  \retval < 0 Error
*/
extern int scip_scip20(serial_t *serial);

/*!
  \brief Switch off the laser, stop measuring

  If the purpose of sending QT command is to stop the reply of MD command, then send the QT without waiting for the MD response to complete. Reply of QT command will be done in urg_receiveData()

  \param serial [i/o] Structure of serial control
  \param return_code [i] Response from the QT command
  \param wait_reply [i] When it does not not wait for response "ScipNoWaitReply" / When waits for response "ScipWaitReply"

  \retval 0 Normal
  \retval < 0 Error
*/
extern int scip_qt(serial_t *serial, int *return_code, int wait_reply);


/*!
  \brief Receive parameter information

  \param serial [i/o] Struct of serial control
  \param parameters [o] urg_parameter_t ,struct member

  \retval 0 Normal
  \retval < 0 Error

*/
extern int scip_pp(serial_t *serial, urg_parameter_t *parameters);


/*!
  \brief Obtain version information

  \param serial [i/o] Struct of serial control
  \param lines [o]    Version information before storing
  \param lines_max [i] Maximum numbers of lines

  \retval 0 Normal
  \retval < 0 Error
*/
extern int scip_vv(serial_t *serial, char *lines[], int lines_max);


/*!
  \brief Change baudrate

  \param serial [i/o] Struct of serial control
  \param baudrate [i] New baudrate value

  \retval 0 Normal
  \retval < 0 Error
*/
extern int scip_ss(serial_t *serial, long baudrate);

#ifdef __cplusplus
}
#endif

#endif /* !C_SCIP_HANDLER_H */
