/*!
  \file
  \brief Serial Communication (Linux, Mac implementation)

  Serial Communication Interface 


  \author Satofumi KAMIMURA

  $Id$
*/

#include <unistd.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>

#include <urg/serial_ctrl.h>
#include <urg/serial_errno.h>

#include <ctype.h>

enum {
  InvalidFd = -1,
};


/* Connection */
int serial_connect(serial_t *serial, const char *device, long baudrate) {

  int flags = 0;
  int ret = 0;

#ifndef MAC_OS
  enum { O_EXLOCK = 0x0 }; /* This cannot be used in LINUX, hence create dummy*/
#endif
  serial->fd_ = open(device, O_RDWR | O_EXLOCK | O_NONBLOCK | O_NOCTTY);
  if (serial->fd_ < 0) {
    /* Failure in connection */
    strerror_r(errno, serial->error_string_, SerialErrorStringSize);

    /* !!! For unknown reason erro -"remove below line" pops up, and hence below line is removed */ 
    /* serial->errno = SerialConnectionFail; */

    return SerialConnectionFail;
  }

  flags = fcntl(serial->fd_, F_GETFL, 0);
  fcntl(serial->fd_, F_SETFL, flags & ~O_NONBLOCK);

  /* Intialization of serial communication */
  tcgetattr(serial->fd_, &serial->sio_);
  serial->sio_.c_iflag = 0;
  serial->sio_.c_oflag = 0;
  serial->sio_.c_cflag &= ~(CSIZE | PARENB | CSTOPB);
  serial->sio_.c_cflag |= CS8 | CREAD | CLOCAL;
  serial->sio_.c_lflag &= ~(ICANON | ECHO | ISIG | IEXTEN);

  serial->sio_.c_cc[VMIN] = 0;
  serial->sio_.c_cc[VTIME] = 0;

  /* Changing baudrate */
  ret = serial_setBaudrate(serial, baudrate);
  if (ret < 0) {
    return ret;
  }

  /* Initialization of serial control struct	 */
  serial->last_ch_ = '\0';

  return 0;
}


/* Disconnection */
void serial_disconnect(serial_t *serial) {

  if (serial->fd_ >= 0) {
    close(serial->fd_);
    serial->fd_ = InvalidFd;
  }
}


int serial_isConnected(serial_t *serial) {

  return (serial->fd_ == InvalidFd) ? 0 : 1;
}


/* Baudrate change */
int serial_setBaudrate(serial_t *serial, long baudrate) {

  long baudrate_value = -1;

  switch (baudrate) {
  case 4800:
    baudrate_value = B4800;
    break;

  case 9600:
    baudrate_value = B9600;
    break;

  case 19200:
    baudrate_value = B19200;
    break;

  case 38400:
    baudrate_value = B38400;
    break;

  case 57600:
    baudrate_value = B57600;
    break;

  case 115200:
    baudrate_value = B115200;
    break;

  default:
#if 0
    /* !!! Buffer is introduced to process  */
    snprintf(serial->error_message, SerialErrorStringSize,
             "No handle baudrate value: %ld", baudrate);
#endif
    return SerialSetBaudrateFail;
  }

  /* Change baudrate*/
  cfsetospeed(&serial->sio_, baudrate_value);
  cfsetispeed(&serial->sio_, baudrate_value);
  tcsetattr(serial->fd_, TCSADRAIN, &serial->sio_);

  return 0;
}


/* Send */
int serial_send(serial_t *serial, const char *data, int data_size) {

  /* !!! Check whether connection is established or not */

#if 0
  int i;
  fprintf(stderr, "[");
  for (i = 0; i < data_size; ++i) {
    if (isprint(data[i])) {
      fprintf(stderr, "%c", data[i]);
    } else {
      fprintf(stderr, "(%02x)", (unsigned char)data[i]);
    }
  }
  fprintf(stderr, "]\n");
#endif

  return write(serial->fd_, data, data_size);
}


/* Receive */
int serial_recv(serial_t *serial, char* data, int data_size_max, int timeout) {

  fd_set rfds;
  struct timeval tv;
  size_t filled;
  int read_n;
  int require_n;

  if (data_size_max <= 0) {
    return 0;
  }

  /* if character written back to buffer is available, then add that character */
  filled = 0;
  if (serial->last_ch_ != '\0') {
    data[0] = serial->last_ch_;
    serial->last_ch_ = '\0';
    ++filled;
  }

  for (; filled < data_size_max; ++filled) {

    /* Set timeout */
    FD_ZERO(&rfds);
    FD_SET(serial->fd_, &rfds);

    tv.tv_sec = timeout / 1000;
    tv.tv_usec = (timeout % 1000) * 1000;

    if (select(serial->fd_ + 1, &rfds, NULL, NULL, &tv) <= 0) {
      /* Timeout occur */
      break;
    }
    require_n = data_size_max - filled;
    read_n = read(serial->fd_, &data[filled], require_n);
    if (read_n <= 0) {
      /* Read error. Stop receiving and return */
      break;
    }
    filled += read_n;
  }

#if 0
  int i;
  fprintf(stderr, "<");
  for (i = 0; i < filled; ++i) {
    if (isprint(data[i])) {
      fprintf(stderr, "%c", data[i]);
    } else {
      fprintf(stderr, "(%02x)", (unsigned char)data[i]);
    }
  }
  fprintf(stderr, ">\n");
#endif

  return filled;
}


/* Put back the "ch" character into the queue*/
void serial_ungetc(serial_t *serial, char ch) {

  serial->last_ch_ = ch;
}
